#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# TODO
# - (WIP) try to identify bottlenecks/optimization points in vector calcs
#
"""
This module was taken from https://github.com/tedyapo/loopfield

See below for the license (MIT) that was attached to it.



Copyright (c) 2016 Theodore C. Yapo

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be
included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
"""

import math
import numpy as np
import scipy.special

nm = 1.e-9
um = 1.e-6
mm = 1.e-3
cm = 1.e-2
m = 1.
uA = 1.e-6
mA = 1.e-3
A = 1.
km = 1.e3
nT = 1.e-9
uT = 1.e-6
mT = 1.e-3
T = 1.

def normalize(v):
	l = np.linalg.norm(v)
	epsilon = 1e-9
	if l < epsilon:
		raise ValueError("Vector with |v| < e normalized")
	return v / l

# filamentary current loop
class Loop(object):
	def __init__(self, position, normal, radius, current):
		self.p = np.array(position, dtype = np.float64)
		self.n = normalize(np.array(normal, dtype = np.float64))
		self.r = np.float64(radius)
		self.i = np.float64(current)

# vector magnetic field (B) produced by collection of current loops
class Field(object):
	def __init__(self,
			length_units = m,
			current_units = A,
			field_units = T):
		self.loops = []
		self._length_units = length_units
		self._field_units = field_units
		self._epsilon = np.finfo(np.float64).eps
	
	def addLoop(self, loop):
		self.loops.append(loop)
	
	def evaluate(self, position):
		_p = np.atleast_2d(position)
		B = np.zeros(_p.shape)
		for loop in self.loops:
			B += self._evalLoop(_p, loop)
		return np.squeeze(B / self._field_units)
	
	def get_rvect(self, p, loop):
		r_vect = (p - loop.p) * self._length_units
		r = np.linalg.norm(r_vect, axis=1, keepdims=True)
		z = r_vect.dot(loop.n.T)
		rho_vect = r_vect - np.outer(z, loop.n)
		rho = np.linalg.norm(rho_vect, axis=1)
		idxLargeRho = rho > self._epsilon
		rho_vect[idxLargeRho,] = \
		  (rho_vect[idxLargeRho,].T/rho[idxLargeRho]).T
		return rho, z, rho_vect
	def get_k2mats(self, p, loop, rho, z):
		a = loop.r * self._length_units
		alpha2 = a*a + rho*rho + z*z - 2.*a*rho
		beta2 = a*a + rho*rho + z*z + 2.*a*rho
		beta = np.sqrt(beta2)        
		c = 4.e-7 * loop.i  # \mu_0  I / \pi
		a2b2 = alpha2 / beta2
		Ek2 = scipy.special.ellipe(1. - a2b2)
		Kk2 = scipy.special.ellipkm1(a2b2)
		return a, alpha2, beta, c, Ek2, Kk2
	def get_brho(self, p, loop, alpha2, beta, rho, z, a, c, Ek2, Kk2):
		denom = (2. * alpha2 * beta * rho)
		with np.errstate(invalid='ignore'):
			numer = c*z*((a*a + rho*rho + z*z)*Ek2 - alpha2*Kk2)
		sw = np.abs(denom) > self._epsilon
		Brho = np.zeros(numer.shape)
		Brho[sw] = numer[sw] / denom[sw]
		return Brho
	def get_bz(self, p, loop, alpha2, beta, rho, z, a, c, Ek2, Kk2):
		denom = (2. * alpha2 * beta)
		with np.errstate(invalid='ignore'):
			numer = c*((a*a - rho*rho - z*z)*Ek2 + alpha2*Kk2)
		sw = np.abs(denom) > self._epsilon
		Bz = np.full(numer.shape, np.inf)
		Bz[sw] = numer[sw] / denom[sw]
		return Bz
	
	def _evalLoop(self, p, loop):
		## determines r_vect
		#r_vect = (p - loop.p) * self._length_units
		#r = np.linalg.norm(r_vect, axis=1, keepdims=True)
		#z = r_vect.dot(loop.n.T)
		#rho_vect = r_vect - np.outer(z, loop.n)
		#rho = np.linalg.norm(rho_vect, axis=1)
		#idxLargeRho = rho > self._epsilon
		#rho_vect[idxLargeRho,] = \
		#  (rho_vect[idxLargeRho,].T/rho[idxLargeRho]).T
		rho, z, rho_vect = self.get_rvect(p, loop)
		
		## determines Ek2 and Kk2
		#a = loop.r * self._length_units
		#alpha2 = a*a + rho*rho + z*z - 2.*a*rho
		#beta2 = a*a + rho*rho + z*z + 2.*a*rho
		#beta = np.sqrt(beta2)
		#c = 4.e-7 * loop.i  # \mu_0  I / \pi
		#a2b2 = alpha2 / beta2
		#Ek2 = scipy.special.ellipe(1. - a2b2)
		#Kk2 = scipy.special.ellipkm1(a2b2)
		a, alpha2, beta, c, Ek2, Kk2 = self.get_k2mats(p, loop, rho, z)
		
		## determines Brho
		#denom = (2. * alpha2 * beta * rho)
		#with np.errstate(invalid='ignore'):
		#	numer = c*z*((a*a + rho*rho + z*z)*Ek2 - alpha2*Kk2)
		#sw = np.abs(denom) > self._epsilon
		#Brho = np.zeros(numer.shape)
		#Brho[sw] = numer[sw] / denom[sw]
		Brho = self.get_brho(p, loop, alpha2, beta, rho, z, a, c, Ek2, Kk2)
		
		## determines Bz
		#denom = (2. * alpha2 * beta)
		#with np.errstate(invalid='ignore'):
		#	numer = c*((a*a - rho*rho - z*z)*Ek2 + alpha2*Kk2)
		#sw = np.abs(denom) > self._epsilon
		#Bz = np.full(numer.shape, np.inf)
		#Bz[sw] = numer[sw] / denom[sw]
		Bz = self.get_bz(p, loop, alpha2, beta, rho, z, a, c, Ek2, Kk2)
		
		# finally gets the result
		return (Brho * rho_vect.T).T + np.outer(Bz, loop.n)
